/*
 * Copyright (c) 2006 Oracle.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */
#include <linux/kernel.h>
#include <linux/random.h>

#include "rds.h"

/*
 * All of connection management is simplified by serializing it through
 * work queues that execute in a connection managing thread.
 *
 * TCP wants to send acks through sendpage() in response to data_ready(),
 * but it needs a process context to do so.
 *
 * The receive paths need to allocate but can't drop packets (!) so we have
 * a thread around to block allocating if the receive fast path sees an
 * allocation failure.
 */
struct workqueue_struct *rds_wq;
EXPORT_SYMBOL_GPL(rds_wq);

void rds_connect_complete(struct rds_connection *conn)
{
	set_bit(RDS_CONN_CONNECTED, &conn->c_status);
	smp_mb__after_clear_bit();
	conn->c_reconnect_jiffies = 0;
	set_bit(0, &conn->c_map_queued);
	queue_delayed_work(rds_wq, &conn->c_send_w, 0);
	queue_delayed_work(rds_wq, &conn->c_recv_w, 0);
}
EXPORT_SYMBOL_GPL(rds_connect_complete);

/*
 * This random exponential backoff is relied on to eventually resolve racing
 * connects. 
 *
 * If connect attempts race then both parties drop both connections and come
 * here to wait for a random amount of time before trying again.  Eventually
 * the backoff range will be so much greater than the time it takes to
 * establish a connection that one of the pair will establish the connection
 * before the other's random delay fires.
 *
 * Connection attempts that arrive while a connection is already established
 * are also considered to be racing connects.  This lets a connection from
 * a rebooted machine replace an existing stale connection before the transport
 * notices that the connection has failed.
 */
void rds_queue_delayed_reconnect(struct rds_connection *conn)
{
	unsigned long rand;

	/* make sure we don't divide by 0 */
	if (conn->c_reconnect_jiffies == 0)
		conn->c_reconnect_jiffies = rds_sysctl_reconnect_min_jiffies;

	get_random_bytes(&rand, sizeof(rand));
	rdsdebug("%lu delay %lu ceil conn %p for %u.%u.%u.%u -> %u.%u.%u.%u\n",
		 rand % conn->c_reconnect_jiffies, conn->c_reconnect_jiffies,
		 conn, NIPQUAD(conn->c_laddr), NIPQUAD(conn->c_faddr));
	queue_delayed_work(rds_wq, &conn->c_conn_w,
			   rand % conn->c_reconnect_jiffies);

	conn->c_reconnect_jiffies = min(conn->c_reconnect_jiffies * 2,
				        rds_sysctl_reconnect_max_jiffies);
}

void rds_connect_worker(struct work_struct *work)
{
	struct rds_connection *conn = container_of(work, struct rds_connection, c_conn_w.work);
	int ret;

	/* notice if listen has already completed the connection for us */
	if (!test_bit(RDS_CONN_CONNECTED, &conn->c_status) &&
	    !test_and_set_bit(RDS_CONN_CONNECTING, &conn->c_status)) {
		ret = conn->c_trans->conn_connect(conn);
		rdsdebug("connect conn %p for %u.%u.%u.%u -> %u.%u.%u.%u "
			 "ret %d\n", conn, NIPQUAD(conn->c_laddr),
			 NIPQUAD(conn->c_faddr), ret);
		if (ret) {
			clear_bit(RDS_CONN_CONNECTING, &conn->c_status);
			rds_queue_delayed_reconnect(conn);
		}
	}
}

void rds_shutdown_worker(struct work_struct *work)
{
	struct rds_connection *conn = container_of(work, struct rds_connection, c_down_w);
	int was_conn;

	/* shut it down if .conn_connect succeeded */
	was_conn = test_and_clear_bit(RDS_CONN_CONNECTED, &conn->c_status);
	if (test_and_clear_bit(RDS_CONN_CONNECTING, &conn->c_status)) {
		down(&conn->c_send_sem);
		conn->c_trans->conn_shutdown(conn);
		rds_conn_reset(conn);
		up(&conn->c_send_sem);
	}

	/* then reconnect if it's still live */
	cancel_delayed_work(&conn->c_conn_w);
	if (!hlist_unhashed(&conn->c_hash_node)) {
		if (!was_conn)
			rds_queue_delayed_reconnect(conn);
		else
			queue_delayed_work(rds_wq, &conn->c_conn_w, 0);
	}
}

void rds_send_worker(struct work_struct *work)
{
	struct rds_connection *conn = container_of(work, struct rds_connection, c_send_w.work);
	int ret;

	if (test_bit(RDS_CONN_CONNECTED, &conn->c_status)) {
		ret = rds_send_xmit(conn);
		rdsdebug("conn %p ret %d\n", conn, ret);
		switch (ret) {
			case -EAGAIN:
				rds_stats_inc(s_send_immediate_retry);
				queue_delayed_work(rds_wq, &conn->c_send_w, 0);
				break;
			case -ENOMEM:
				rds_stats_inc(s_send_delayed_retry);
				queue_delayed_work(rds_wq, &conn->c_send_w, 2);
			default:
				break;
		}
	}
}

void rds_recv_worker(struct work_struct *work)
{
	struct rds_connection *conn = container_of(work, struct rds_connection, c_recv_w.work);
	int ret;

	if (test_bit(RDS_CONN_CONNECTED, &conn->c_status)) {
		ret = conn->c_trans->recv(conn);
		rdsdebug("conn %p ret %d\n", conn, ret);
		switch (ret) {
			case -EAGAIN:
				rds_stats_inc(s_recv_immediate_retry);
				queue_delayed_work(rds_wq, &conn->c_recv_w, 0);
				break;
			case -ENOMEM:
				rds_stats_inc(s_recv_delayed_retry);
				queue_delayed_work(rds_wq, &conn->c_recv_w,
						   2);
			default:
				break;
		}
	}
}

void __exit rds_threads_exit(void)
{
	destroy_workqueue(rds_wq);
}

int __init rds_threads_init(void)
{
	rds_wq = create_singlethread_workqueue("krdsd");
	if (rds_wq == NULL) 
		return -ENOMEM;

	return 0;
}
